<?php

namespace App\Services;

use App\Api\OpenCartClient;
use App\supplier1\SupplierOneAdapter;
use Exception;


class ProductService {
    private $client;
    private $supplierAdapter;
    private $logManager;
    private $batchSize = 40; // Process products in batches of 300

    public function __construct() {
        $this->client = new OpenCartClient();
        $this->supplierAdapter = new SupplierOneAdapter();
        $this->logManager = new LogManager();
    }

    /**
     * Delete products that exist on the website but not in the XML feed.
     * This function fetches all products from OpenCart and compares them with the current XML data.
     * If a product exists on OpenCart but not in the XML, it is marked for deletion.
     * 
     * @param array $xmlProducts An array of SKUs from the XML feed to compare against OpenCart.
     * @return array The results of the deletion requests, showing deleted product IDs or errors if any.
     */
    public function deleteObsoleteProducts() {
        try {
            $xmlProducts = $this->supplierAdapter->fetchProducts(); // Fetch products from the XML feed

            // Extract codes (SKUs) from the XML feed
            $xmlSkus = array_column($xmlProducts, 'code');

            // Step 1: Fetch current products from OpenCart
            $currentProducts = $this->client->getProductData();

            // Step 2: Extract SKUs and locations from OpenCart product data for comparison
            $openCartSkus = [];
            foreach ($currentProducts as $product) {
                $openCartSkus[$product['sku']] = [
                    'product_id' => $product['product_id'],
                    'location' => $product['location']
                ];
            }

            // Step 3: Identify products to delete (in OpenCart but not in XML, and location does not contain "notdelete")
            $productIdsToDelete = [];
            foreach ($openCartSkus as $sku => $details) {
                if (!in_array($sku, $xmlSkus) && strpos($details['location'], 'notdelete') === false) {
                    $productIdsToDelete[] = $details['product_id'];
                }
            }

            // Step 4: Delete the identified products
            if (!empty($productIdsToDelete)) {
                $deletionResults = $this->client->deleteProducts($productIdsToDelete);
                $this->logManager->logInfo("Deleted " . count($productIdsToDelete) . " obsolete products.", 'productservice', 'supplier1');
                return $deletionResults;
            } else {
                $this->logManager->logInfo("No obsolete products found to delete.", 'productservice', 'supplier1');
                return ['status' => 'No deletions necessary'];
            }
        } catch (Exception $e) {
            $this->logManager->logError('Failed to delete obsolete products: ' . $e->getMessage(), 'productservice', 'supplier1');
            return ['error' => $e->getMessage()];
        }
    }

    /**
     * Process and sync products from XML feed with OpenCart in bulk.
     */
    public function processProducts() {
        try {
            $products = $this->supplierAdapter->fetchProducts(); // Fetch products from the XML feed
            $this->logManager->logError('Product fetch is done ', 'productservice', 'supplier1');

            // Split products into batches
            $productBatches = array_chunk($products, $this->batchSize);

            // Iterate through each batch of products and sync
            foreach ($productBatches as $batch) {
                $this->syncProductsInBatch($batch);
            }
        } catch (Exception $e) {
            $this->logManager->logError('Failed to process products: ' . $e->getMessage(), 'productservice', 'supplier1');
        }
    }

    /**
     * Sync a batch of products with OpenCart.
     * @param array $products Array of products in a batch
     */
    private function syncProductsInBatch(array $products) {
        $productsToAdd = [];
        $productsToEdit = [];

        try {
            // Step 1: Collect all SKUs
            $skus = array_column($products, 'code');

            // Step 2: Get all product IDs by SKUs asynchronously
            $skuToProductIdMap = $this->client->getProductIdsBySku($skus);

            // Step 3: Process each product with the product ID map
            foreach ($products as $product) {
                // Get the product ID from the map
                $productId = $skuToProductIdMap[$product['code']];

                // Prepare the product data
                $productData = $this->prepareProductData($product);

                if ($productId) {
                    // If the product exists, add it to the products to edit
                    $productData['product_id'] = $productId; // Add product ID for editing
                    $productsToEdit[] = $productData;
                } else {
                    // If the product doesn't exist, add it to the products to add
                    $productsToAdd[] = $productData;
                }
            }

            // Step 4: Send the products in bulk to OpenCart API
            $this->bulkSyncProducts($productsToAdd, $productsToEdit);

        } catch (Exception $e) {
            $this->logManager->logError('Failed to sync products batch: ' . $e->getMessage(), 'productservice', 'supplier1');
        }
    }

    /**
     * Bulk sync products with OpenCart by adding or editing them in batches.
     * @param array $productsToAdd Array of products to add
     * @param array $productsToEdit Array of products to edit
     */
    private function bulkSyncProducts(array $productsToAdd, array $productsToEdit) {
        try {
            // Bulk add new products
            if (!empty($productsToAdd)) {
                $this->client->bulkAddProducts($productsToAdd);
                $this->logManager->logInfo("Bulk added " . count($productsToAdd) . " products", 'productservice', 'supplier1');
            }

            // Bulk edit existing products
            if (!empty($productsToEdit)) {
                $this->client->bulkEditProducts($productsToEdit);
                $this->logManager->logInfo("Bulk edited " . count($productsToEdit) . " products", 'productservice', 'supplier1');
            }
        } catch (Exception $e) {
            $this->logManager->logError('Failed to bulk sync products: ' . $e->getMessage(), 'productservice', 'supplier1');
        }
    }

    /**
     * Prepare product data for OpenCart API
     * @param array $product
     * @return array
     */
    private function prepareProductData(array $product) {
        // Default category handling logic
       $categoryId = null;
       
       // Check for special conditions based on `index_entry` or "Environmental" category
       if (isset($product['index_entry']) && !empty($product['index_entry'])) {
           // If index_entry is "Environmental", handle it differently
           if (strpos(strtolower($product['index_entry']), 'environmental') !== false) {
               // Handle environmental category differently, for example assigning a special category
               $categoryId = $this->getCategoryIdByName('Environmental');
               
               if (!$categoryId) {
                   $categoryId = $_ENV['DEFAULT_ENVIRONMENTAL_CATEGORY_ID'] ?? 0; // Use default environmental category
                   $this->logManager->logError('Environmental category not found for product: ' . $product['code'] . '. Using default environmental category.', 'productservice', 'supplier1');
               }
           } else {
               // For other `index_entry` values, attempt to fetch the category by name
               $categoryId = $this->getCategoryIdByName($product['index_entry']);
           }
       }

       // Fallback to main category if index_entry does not trigger special conditions
       if (!$categoryId) {
           if (isset($product['category']) && !empty($product['category'])) {
               $categoryId = $this->getCategoryIdByName($product['category']);
           } else {
               $categoryId = $_ENV['DEFAULT_CATEGORY_ID'] ?? 0; // Fallback to a default category if none is provided
               $this->logManager->logError('No valid category or index_entry for product: ' . $product['code'] . '. Using default category.', 'productservice', 'supplier1');
           }
       }

       // Prepare SEO keyword, using the product title or code as an example
       $seoKeyword = $this->generateSeoUrl($product['title'], $product['code']);

       // Check if there are personalisation methods
       $hasPersonalisation = false;
       $lowestQuantityBreak = null;

       foreach ($product['quantity_breaks'] as $quantityBreak) {
            // Check for personalisation methods
            if (!$hasPersonalisation && !empty($quantityBreak['personalisation_methods'])) {
                $hasPersonalisation = true;
            }

            // Determine the lowest quantity break
            if ($lowestQuantityBreak === null || $quantityBreak['quantity'] < $lowestQuantityBreak['quantity']) {
                $lowestQuantityBreak = $quantityBreak;
            }
       }

        // Validation: Check if minimum quantity matches the lowest quantity break
        if ($lowestQuantityBreak && $product['minimum_quantity'] != $lowestQuantityBreak['quantity']) {
            // Log the product with quantity issue
            $this->logManager->logError(
                'Quantity mismatch: Product "' . $product['title'] . '" (' . $product['code'] . ') has a minimum quantity (' . $product['minimum_quantity'] . ') that does not match the lowest quantity break (' . $lowestQuantityBreak['quantity'] . ').',
                'productmismatch',
                'supplier1'
            );
        }


       $productData = [
           'model' => $product['code'],
           'quantity' => 1000, // Example default
           'price' => $this->getBasePrice($product), // Base price from lowest selling price
           'tax_class_id' => 1, // Default tax class
           'manufacturer_id' => $product['supplier_id'],
           'sku' => $product['code'],
           'status' => 1,
           'minimum' => $product['minimum_quantity'],
           'image' => $product['image'],
           'product_description' => [
               [
                   'language_id' => 1,
                   'name' => $product['title'],
                   'description' => $product['description'],
                   'meta_title' => $product['title'] . ' - ' . $product['code'],
                   'meta_description' => $product['description'],
                   'meta_keyword' => $product['title']
               ]
           ],
           'product_category' => [$categoryId],
           'product_attribute' => $this->prepareProductAttributes($product), // Attributes for printing methods and pricing without color info
           'product_seo_url' => [
               [
                   'keyword' => $seoKeyword,
                   'language_id' => 1,
                   'store_id' => 0
               ]
           ]
       ];

        // If the product has personalisation methods, add options and option discounts
       if ($hasPersonalisation) {
           $productData['product_option'] = $this->prepareProductOptions($product);
           $productData['product_discount'] = [];
       } else {
           // No personalisation methods, use product-level discounts
           $productData['product_option'] = [];
           $productData['product_discount'] = $this->prepareProductDiscounts($product);
       }

       return $productData;
   }

   /**
     * Generate a SEO-friendly URL by replacing and removing specific characters.
     *
     * @param string $title The product title.
     * @param string $code The product code (appended to title for uniqueness).
     * @return string SEO-friendly URL.
     */
    private function generateSeoUrl(string $title, string $code): string {
        // Replace specific characters
        $replacements = [
            '&' => 'and',
            '£' => 'pound',
            '€' => 'euro',
            '@' => 'at',
            '%' => 'percent',
            '#' => '',
            '®' => 'r',
            '™' => 'tm',
            '+' => 'plus',
        ];

        // Replace unwanted characters based on the mapping
        $seoTitle = strtr($title, $replacements);

        // Remove any remaining unwanted characters except dashes
        $seoTitle = preg_replace('/[^a-zA-Z0-9\- ]/', '', $seoTitle);

        // Replace whitespace and multiple dashes with a single dash
        $seoTitle = preg_replace('/\s+/', '-', $seoTitle);
        $seoTitle = preg_replace('/-+/', '-', $seoTitle);

        // Convert to lowercase
        $seoTitle = strtolower(trim($seoTitle, '-'));

        // Append the product code to ensure uniqueness
        return $seoTitle . '-' . strtolower($code);
    }


   /**
    * Get the base price from the lowest selling price of the product
    * @param array $product
    * @return float
    */
   private function getBasePrice(array $product) {
       $lowestPrice = PHP_FLOAT_MAX;

       foreach ($product['quantity_breaks'] as $quantityBreak) {
           // If personalisation methods are available, use them to find the lowest price
           if (!empty($quantityBreak['personalisation_methods'])) {
               foreach ($quantityBreak['personalisation_methods'] as $method) {
                   foreach ($method['personalisations'] as $personalisation) {
                       if ($personalisation['selling_price'] < $lowestPrice) {
                           $lowestPrice = $personalisation['selling_price'];
                       }
                   }
               }
           } else {
               // If no personalisation methods, use the selling price of the quantity break
               if ($quantityBreak['selling_price'] < $lowestPrice) {
                   $lowestPrice = $quantityBreak['selling_price'];
               }
           }
       }

       return $lowestPrice;
   }

   /**
    * Prepare product discounts based on quantity breaks (for products without personalisation)
    * @param array $product
    * @return array
    */
   private function prepareProductDiscounts(array $product) {
       $discounts = [];

       foreach ($product['quantity_breaks'] as $quantityBreak) {
           if ($quantityBreak['quantity'] > 1) {
               $discounts[] = [
                   'name' => $product['title'], // Add the product title as the name
                   'customer_group_id' => 1, // Default customer group
                   'quantity' => $quantityBreak['quantity'],
                   'priority' => 1, // Default priority
                   'price' => $quantityBreak['selling_price'],
                   'date_start' => date('Y-m-d'),
                   'date_end' => '0000-00-00' // No end date
               ];
           }
       }

       return $discounts;
   }

    /**
     * Prepare product options with discount handling for different quantity breaks and color options
     * @param array $product The product data from the XML
     * @return array Prepared product options with discounts and color options
    */
    private function prepareProductOptions(array $product) {
        $options = [];
        $printingMethodOptionId = $_ENV['PRINTING_METHOD_OPTION_ID']; // Option ID for printing method from .env
        $colourOptionId = $_ENV['COLOUR_OPTION_ID']; // Option ID for color from .env

        $optionValuesToAdd = [];

        // Step 1: Collect printing methods from all quantity breaks
        foreach ($product['quantity_breaks'] as $quantityBreak) {
            foreach ($quantityBreak['personalisation_methods'] as $method) {
                foreach ($method['personalisations'] as $personalisation) {
                    $optionValueName = $method['name'] . ' - ' . $personalisation['colours'] . ' Color/s';
                    $optionValuesToAdd[] = [
                        'option_id' => $printingMethodOptionId,
                        'name' => $optionValueName,
                    ];
                }
            }
        }

        // Step 2: Collect colors
        if (!empty($product['colours'])) {
            $tidyColours = $this->tidyColours($product['colours']);
            foreach ($tidyColours as $cleanColor) {
                $optionValuesToAdd[] = [
                    'option_id' => $colourOptionId,
                    'name' => $cleanColor,
                ];
            }
        }

        // Step 3: Send all option values to the API for bulk processing
        // The `addOptionValues` function will handle retrieving or creating option values as needed
        $addedOptionValues = $this->client->addOptionValues($optionValuesToAdd);

        // Step 4: Prepare a lookup map for option value names to IDs
        $optionValueMap = [];
        foreach ($addedOptionValues as $optionValue) {
            $optionValueMap[$optionValue['name']] = $optionValue['option_value_id'];
        }

        // Step 5: Prepare printing method options
        $options[$printingMethodOptionId] = [
            'option_id' => $printingMethodOptionId,
            'type' => 'select',
            'required' => 1,
            'product_option_value' => []
        ];

        // Find the lowest quantity break
        $lowestQuantityBreak = null;
        foreach ($product['quantity_breaks'] as $quantityBreak) {
            if ($lowestQuantityBreak === null || $quantityBreak['quantity'] < $lowestQuantityBreak['quantity']) {
                $lowestQuantityBreak = $quantityBreak;
            }
        }

        // Loop through the lowest quantity break to add printing methods and set base prices
        foreach ($lowestQuantityBreak['personalisation_methods'] as $method) {
            foreach ($method['personalisations'] as $personalisation) {
                $optionValueName = $method['name'] . ' - ' . $personalisation['colours'] . ' Color/s';
                $optionValueId = $optionValueMap[$optionValueName];

                $options[$printingMethodOptionId]['product_option_value'][] = [
                    'option_value_id' => $optionValueId,
                    'price' => $personalisation['selling_price'],
                    'price_prefix' => '=',
                    'quantity' => 1000,
                    'subtract' => 1,
                    'points' => 0,
                    'points_prefix' => '+',
                    'weight' => 0,
                    'weight_prefix' => '+',
                    'discount' => [] // Discounts will be added here
                ];

                $lastIndex = count($options[$printingMethodOptionId]['product_option_value']) - 1;

                // Add discounts from higher quantity breaks
                foreach ($product['quantity_breaks'] as $quantityBreak) {
                    //if ($quantityBreak['quantity'] > $lowestQuantityBreak['quantity']) {
                        foreach ($quantityBreak['personalisation_methods'] as $quantityMethod) {
                            if ($quantityMethod['name'] === $method['name']) {
                                foreach ($quantityMethod['personalisations'] as $quantityPersonalisation) {
                                    if ($quantityPersonalisation['colours'] == $personalisation['colours']) {
                                        $options[$printingMethodOptionId]['product_option_value'][$lastIndex]['discount'][] = [
                                            'customer_group_id' => 1,
                                            'quantity' => $quantityBreak['quantity'],
                                            'priority' => 1,
                                            'price' => $quantityPersonalisation['selling_price'],
                                            'date_start' => date('Y-m-d'),
                                            'date_end' => '0000-00-00'
                                        ];
                                    }
                                }
                            }
                        }
                    //}
                }
            }
        }


        // Step 6: Prepare color options
        if (!empty($product['colours'])) {
            $options[$colourOptionId] = [
                'option_id' => $colourOptionId,
                'type' => 'select',
                'required' => 1,
                'product_option_value' => []
            ];

            $tidyColours = $this->tidyColours($product['colours']);

            foreach ($tidyColours as $cleanColor) {
                $optionValueId = $optionValueMap[$cleanColor];
                $options[$colourOptionId]['product_option_value'][] = [
                    'option_value_id' => $optionValueId,
                    'price' => 0.00,
                    'price_prefix' => '+',
                    'quantity' => 1000,
                    'subtract' => 1,
                    'points' => 0,
                    'points_prefix' => '+',
                    'weight' => 0,
                    'weight_prefix' => '+',
                    'discount' => [] // No discounts for color options
                ];
            }
        }

        return array_values($options); // Return the options as an array without keys
    }

    

    private function tidyColours($colours) {
       $colours = str_replace('\"', '', $colours); // Remove quotes
       $coloursArray = explode(',', $colours);

       $tidyColours = [];
       foreach ($coloursArray as $colour) {
           $colour = trim($colour);
           $colour = strtolower($colour);
           $colour = ucwords($colour);

           // Additional cleanup logic based on original script
           if (preg_match('/Melange/i', $colour)) {
               $colour = preg_replace('/Melange/i', 'Melange', $colour);
           }
           if (preg_match('/blackchrome/i', $colour)) {
               $colour = preg_replace('/blackchrome/', 'black/chrome', $colour);
           }
           if (preg_match('/,/', $colour)) {
               $tidyColours = array_merge($tidyColours, explode(',', $colour));
               continue;
           }
           if (preg_match('/\|/', $colour)) {
               $tidyColours = array_merge($tidyColours, explode('|', $colour));
               continue;
           }
           if (preg_match('/\//', $colour) && !preg_match('/-/', $colour)) {
               $splitColours = explode('/', $colour);
               $splitColours = array_map([$this, 'cleanupColour'], $splitColours);
               $colour = implode("/", $splitColours);
           } elseif (preg_match('/\//', $colour) && preg_match('/-/', $colour)) {
               $splitColours = explode('-', $colour);
               foreach ($splitColours as &$split) {
                   if (preg_match('/\//', $split)) {
                       $split = implode("/", array_map([$this, 'cleanupColour'], explode('/', $split)));
                   }
               }
               $colour = implode("-", $splitColours);
           } elseif (preg_match('/-opaque/i', $colour) || preg_match('/-froste/i', $colour) || preg_match('/-clear/i', $colour)) {
               $splitColours = explode('-', $colour);
               $splitColours = array_map([$this, 'cleanupColour'], $splitColours);
               $colour = implode("-", $splitColours);
           } elseif (preg_match('/[a-z]+-[a-z]+/i', $colour)) {
               $splitColours = explode('-', $colour);
               $splitColours = array_map([$this, 'cleanupColour'], $splitColours);
               $colour = implode(" ", $splitColours);
           } else {
               $colour = $this->cleanupColour($colour);
           }

           $tidyColours[] = $colour;
       }

       return array_unique($tidyColours);
   }

   private function cleanupColour($colour) {
       $colour = ltrim($colour);
       $colour = rtrim($colour);
       $colour = strtolower($colour);
       $colour = ucwords($colour);
       return $colour;
   }


   /**
    * Prepare product attributes for printing methods and pricing
    * Includes primary/secondary categories and dynamic printing methods (up to 4)
    * @param array $product The product data
    * @return array Prepared product attributes
    */
   private function prepareProductAttributes(array $product) {
       $attributes = [];

       // Attribute code map for up to 4 printing methods
       $attributeCodeMap = [
           1 => [
               'printing_method' => $_ENV['ATTRIBUTE_ID_PRINTING_METHOD'],
               'origination' => $_ENV['ATTRIBUTE_ID_ORIGINATION'],
               'repeat_origination' => $_ENV['ATTRIBUTE_ID_REPEAT_ORIGINATION']
           ],
           2 => [
               'printing_method' => $_ENV['ATTRIBUTE_ID_PRINTING_METHOD_SECOND'],
               'origination' => $_ENV['ATTRIBUTE_ID_ORIGINATION_SECOND'],
               'repeat_origination' => $_ENV['ATTRIBUTE_ID_REPEAT_ORIGINATION_SECOND']
           ],
           3 => [
               'printing_method' => $_ENV['ATTRIBUTE_ID_PRINTING_METHOD_THIRD'],
               'origination' => $_ENV['ATTRIBUTE_ID_ORIGINATION_THIRD'],
               'repeat_origination' => $_ENV['ATTRIBUTE_ID_REPEAT_ORIGINATION_THIRD']
           ],
           4 => [
               'printing_method' => $_ENV['ATTRIBUTE_ID_PRINTING_METHOD_FOURTH'],
               'origination' => $_ENV['ATTRIBUTE_ID_ORIGINATION_FOURTH'],
               'repeat_origination' => $_ENV['ATTRIBUTE_ID_REPEAT_ORIGINATION_FOURTH']
           ]
       ];

       // Add primary category attribute
       $attributes[] = [
           'attribute_id' => $_ENV['ATTRIBUTE_ID_PRIMARY_CATEGORY'],
           'product_attribute_description' => [
               [
                   'language_id' => 1,
                   'text' => $product['category']
               ]
           ]
       ];

       // Add secondary category attribute if available
       if (!empty($product['index_entry'])) {
           $attributes[] = [
               'attribute_id' => $_ENV['ATTRIBUTE_ID_SECONDARY_CATEGORY'],
               'product_attribute_description' => [
                   [
                       'language_id' => 1,
                       'text' => $product['index_entry']
                   ]
               ]
           ];
       }

       // Get distinct printing methods from the product data
       $printingMethods = [];
       foreach ($product['quantity_breaks'] as $quantityBreak) {
           foreach ($quantityBreak['personalisation_methods'] as $method) {
               // Check if a method with the same name already exists
               if (!isset($printingMethods[$method['name']])) {
                   $printingMethods[$method['name']] = $method; // Store by method name to ensure uniqueness
               }
           }
       }

       // Add attributes for up to 4 printing methods
       $methodCount = 0;
       foreach ($printingMethods as $method) {
           if ($methodCount >= 4) {
               break; // Only process up to 4 printing methods
           }
           
           $map = $attributeCodeMap[$methodCount + 1]; // Get the correct map for the method index (1-based)

           // Add printing method attribute
           $attributes[] = [
               'attribute_id' => $map['printing_method'],
               'product_attribute_description' => [
                   [
                       'language_id' => 1,
                       'text' => $method['name']
                   ]
               ]
           ];

           // Add origination cost attribute
           $attributes[] = [
               'attribute_id' => $map['origination'],
               'product_attribute_description' => [
                   [
                       'language_id' => 1,
                       'text' => '£' . $method['origination']
                   ]
               ]
           ];

           // Add repeat origination cost attribute
           $attributes[] = [
               'attribute_id' => $map['repeat_origination'],
               'product_attribute_description' => [
                   [
                       'language_id' => 1,
                       'text' => '£' . $method['repeat_origination']
                   ]
               ]
           ];

           $methodCount++;
       }

       return $attributes;
   }

   /**
    * Get category ID by name
    * @param string $categoryName
    * @return int
    */
   private function getCategoryIdByName(string $categoryName) {
       return $this->client->getCategoryIdByName($categoryName) ?? $_ENV['DEFAULT_CATEGORY_ID'];
   }
}
