<?php

namespace App\supplier1;

use App\Services\LogManager;
use App\Services\DownloadService;

class SupplierOneAdapter {

    private $logManager;
    private $imageDownloader;
    private $localFilePath;
    private $downloadService;

    public function __construct() {
        $this->logManager = new LogManager();
        $this->imageDownloader = new ImageDownloader();
        $this->downloadService = new DownloadService();
        $this->localFilePath = __DIR__ . '/../../data/supplier1/supplier_one.xml';
    }

    public function fetchProducts() {
        if (!file_exists($this->localFilePath)) {
            $this->logManager->logInfo("Local XML file not found, downloading...", 'fetch', 'supplier1');
            $this->downloadService->downloadXml();
        }
    
        if (!file_exists($this->localFilePath)) {
            $this->logManager->logError("Failed to download the XML file.", 'fetch', 'supplier1');
            throw new \Exception("Failed to download the XML file.");
        }
    
        // Use the new streaming parser
        return $this->parseXml($this->localFilePath);
    }

    private function parseXml($xmlFilePath) {
        $reader = new \XMLReader();
    
        // Open the XML file
        if (!$reader->open($xmlFilePath)) {
            $this->logManager->logError("Failed to open XML file: $xmlFilePath", 'parse', 'supplier1');
            throw new \Exception("Failed to open XML file: $xmlFilePath");
        }
    
        $products = [];
    
        // Loop through the XML nodes
        while ($reader->read()) {
            // Check if the node is an element and matches the 'product' node
            if ($reader->nodeType == \XMLReader::ELEMENT && $reader->localName == 'product') {
                // Convert the current node to a SimpleXMLElement
                $node = $reader->expand();
                $productXml = simplexml_import_dom($node);
    
                // Parse the product data
                $product = [
                    'code' => (string) $productXml->code,
                    'title' => (string) $productXml->title,
                    'description' => (string) $productXml->description,
                    'last_updated' => (int) $productXml->last_updated,
                    'colours' => (string) $productXml->colours,
                    'supplier' => (string) $productXml->supplier,
                    'supplier_id' => (int) $productXml->supplier_id,
                    'category' => (string) $productXml->category,
                    'index_entry' => (string) $productXml->index_entry,
                    'dimensions' => (string) $productXml->dimensions,
                    'lead_time' => (int) $productXml->lead_time,
                    'express_lead_time' => (int) $productXml->express_lead_time,
                    'carton_quantity' => (int) $productXml->carton_quantity,
                    'carton_weight' => (float) $productXml->carton_weight,
                    'package_type' => (string) $productXml->package_type,
                    'catalogue_page' => (int) $productXml->catalogue_page,
                    'minimum_quantity' => (int) $productXml->minimum_quantity,
                    'prices_per_thousand' => (int) $productXml->prices_per_thousand,
                    'image' => (string) $productXml->image,
                    'quantity_breaks' => $this->parseQuantityBreaks($productXml->quantity_breaks),
                ];
    
                $products[] = $product;
            }
        }
    
        // Close the XML reader
        $reader->close();
    
        return $products;
    }    

    private function parseQuantityBreaks($quantityBreaksNode) {
        return array_map(function($breakNode) {
            return [
                'quantity' => (int) $breakNode->quantity,
                'plain_cost' => (float) $breakNode->plain_cost,
                'carriage_cost' => (float) $breakNode->carriage_cost,
                'selling_price' => (float) $breakNode->selling_price,
                'personalisation_methods' => $this->parsePersonalisationMethods($breakNode->personalisation_methods)
            ];
        }, iterator_to_array($quantityBreaksNode->quantity_break));
    }    

    private function parsePersonalisationMethods($methodsNode) {
        return array_map(function($methodNode) {
            return [
                'name' => (string) $methodNode->name,
                'origination' => (float) $methodNode->origination,
                'repeat_origination' => (float) $methodNode->repeat_origination,
                'personalisation_area' => (string) $methodNode->personalisation_area,
                'personalisation_position' => (string) $methodNode->personalisation_position,
                'minimum_colours' => (int) $methodNode->minimum_colours,
                'maximum_colours' => (int) $methodNode->maximum_colours,
                'first_colour_cost' => (float) $methodNode->first_colour_cost,
                'extra_colour_cost' => (float) $methodNode->extra_colour_cost,
                'personalisations' => $this->parsePersonalisations($methodNode->personalisations)
            ];
        }, iterator_to_array($methodsNode->personalisation_method));
    }
    
    private function parsePersonalisations($personalisationsNode) {
        return array_map(function($personalisationNode) {
            return [
                'colours' => (int) $personalisationNode->colours,
                'selling_price' => (float) $personalisationNode->selling_price,
            ];
        }, iterator_to_array($personalisationsNode->personalisation));
    }
    
}
